define(['angular', 'angularMocks', 'AssessmentBuilderEditController', 'angularUiBootstrap'], function(angular, mocks) {
    'use strict';

    describe("The Assessment Builder Edit controller", function (){
        var controller, 
            scope,
            locationMock, 
            stateMock,
            stateParamsMock,
			focusServiceMock,
            previousStateMock,
            globalDelayTrackerMock,
            modalMock,
            modalServiceMock,
            assessmentServiceMock;

        beforeEach( function() {
            module('angularTemplateApp');

            locationMock = jasmine.createSpyObj('$location', ['path']);
            stateMock = jasmine.createSpyObj('$state', ['go']);
            stateParamsMock = {
                id: "123",
                assessment: {
                    uniqueTitle: "title",
                    purpose: "",
                    healthAssessment: true,
                    waitingPeriod: 0,
                    notes: "",
                    version: 1,
                    authenticationStrategy: "BASIC",
                    scoringAlgorithm: "UNSCORED",
                    publish: false,
                    status: "Draft",
                    createdDate: "12/12/12 12:12:12", 
                    lastModifiedDate: "12/12/12 12:12:12",
                    question: []
                }
            }
			focusServiceMock = jasmine.createSpyObj('focusService', ['focusMain', 'focusElement']);
            previousStateMock = {};
            previousStateMock.isFirstAppRoute = true;
            globalDelayTrackerMock = jasmine.createSpyObj('globalDelayTracker', ['add', 'remove', 'active']);
            globalDelayTrackerMock.active.andCallFake(function(){
                return true;
            });
            modalMock = jasmine.createSpyObj('$modal', ['open']);
            modalMock.open.andCallFake(function(options) {
                for(var index in options.resolve) {
                    options.resolve[index](); 
                }
                return {
                    result: {
                        finally: function(callback) {
                            callback(); 
                        },
                        then: function(successCb, failCb) {
                            successCb({data: "good response"}); 
                            failCb({data: "bad response"})
                        }
                    }
                };
            }); 
            modalServiceMock = jasmine.createSpyObj('modalService', ['showModal']);
            modalServiceMock.showModal.andCallFake(function() {
                return {
                    then: function(successCb, failCb) {
                        successCb({data: "good response"}); 
                        if(failCb) {
                            failCb({data: "bad response"}); 
                        }
                    }
                };
            });
            assessmentServiceMock = jasmine.createSpyObj('assessmentService', ['getAssessments', 'saveAssessment', 'deleteAssessment', 'getAssessmentById', 'getModel', 'getBlankAssessment', 'inactivateAssessment', 'modifyAssessment']);
            assessmentServiceMock.getAssessments.andCallFake(function() {
                var firstAssessment = angular.copy(stateParamsMock.assessment); 
                firstAssessment.status = "Active";
                var secondAssessment = angular.copy(stateParamsMock.assessment);
                secondAssessment.status = "Draft";
                secondAssessment.version = 2;
                var thirdAssessment = angular.copy(stateParamsMock.assessment);

                return {
                    then: function(callback) {
                        callback({
                            data: {
                                assessment: [firstAssessment, secondAssessment, thirdAssessment]
                            }
                        });
                    }
                };
            });
            assessmentServiceMock.saveAssessment.andCallFake(function() {
                return {
                    then: function(successCb, failCb) {
                        successCb({data: "good response"});
                    }
                };
            });
            assessmentServiceMock.deleteAssessment.andCallFake(function() {
                return {
                    then: function(callback) {
                        callback(); 
                    }
                };
            });
            assessmentServiceMock.inactivateAssessment.andCallFake(function() {
                return {
                    then: function(callback) {
                        callback(); 
                    }
                };
            });
            assessmentServiceMock.modifyAssessment.andCallFake(function() {
                return {
                    then: function(callback) {
                        callback({data: {id: '123'}}); 
                    }
                };
            });
            assessmentServiceMock.getModel.andCallFake(function() {
                return {test: 'data'}; 
            });
            assessmentServiceMock.getBlankAssessment.andCallFake(function() {
                return {
                    uniqueTitle: "",
                    purpose: "",
                    healthAssessment: true,
                    waitingPeriod: 0,
                    notes: "",
                    version: 1,
                    authenticationStrategy: "BASIC",
                    scoringAlgorithm: "UNSCORED",
                    publish: false,
                    status: "Draft",
                    createdDate: null, 
                    lastModifiedDate: null,
                    question: []
                };
            });

            module(function($provide){
                $provide.value('$location', locationMock);
                $provide.value('$state', stateMock); 
				$provide.value('focusService', focusServiceMock);
                $provide.value('previousState', previousStateMock);
                $provide.value('globalDelayTracker', globalDelayTrackerMock);
                $provide.value('$modal', modalMock); 
                $provide.value('modalService', modalServiceMock);
                $provide.value('assessmentService', assessmentServiceMock); 
            });

            inject(function($controller, $rootScope) {
                scope = $rootScope.$new();
                controller = $controller('AssessmentBuilderEditController', {$scope:scope, $stateParams: stateParamsMock});
            });
        });

        describe("when the page is loaded", function(){
            it("should update the previous state and focus main when delay tracker resolves", function() {
                globalDelayTrackerMock.active = function() { return false; };
                scope.$apply();
                expect(focusServiceMock.focusMain).toHaveBeenCalled(); 
                expect(previousStateMock.isFirstAppRoute).toBe(false);
            });

            it("should use the assessment from stateparams if it can", function() {
                expect(scope.assessment).toEqual(stateParamsMock.assessment); 
            });

            describe("when there is not an assessment in stateparams, but an id", function() {
                it("should retrieve the assessment and store if it is a success", function() {
                    assessmentServiceMock.getAssessmentById.andCallFake(function() {
                        return {
                            then: function(successCb, failCb) {
                                successCb({status: 200, data: stateParamsMock.assessment}); 
                            }
                        }
                    });
                    inject(function($controller, $rootScope) {
                        scope = $rootScope.$new();
                        controller = $controller('AssessmentBuilderEditController', {$scope:scope, $stateParams: {id: '123', assessment: null}});
                    });
                    expect(scope.assessment).toEqual(stateParamsMock.assessment); 
                });

                it("should retrieve the assessment and store if it is a success.  It should also check if there is an existing draft.", function() {
                    stateParamsMock.assessment.status = "Active"; 
                    assessmentServiceMock.getAssessmentById.andCallFake(function() {
                        return {
                            then: function(successCb, failCb) {
                                successCb({status: 200, data: stateParamsMock.assessment}); 
                            }
                        }
                    });
                    inject(function($controller, $rootScope) {
                        scope = $rootScope.$new();
                        controller = $controller('AssessmentBuilderEditController', {$scope:scope, $stateParams: {id: '123', assessment: null}});
                    });
                    expect(scope.assessment).toEqual(stateParamsMock.assessment); 
                });

                it("should redirect if the id retrieves nothing", function() {
                    assessmentServiceMock.getAssessmentById.andCallFake(function() {
                        return {
                            then: function(successCb, failCb) {
                                successCb({status: 204}); 
                            }
                        }
                    });
                    inject(function($controller, $rootScope) {
                        scope = $rootScope.$new();
                        controller = $controller('AssessmentBuilderEditController', {$scope:scope, $stateParams: {id: '123', assessment: null}});
                    });
                    expect(stateMock.go).toHaveBeenCalled(); 
                });

                it("should redirect if the id fails to retrieve", function() {
                    assessmentServiceMock.getAssessmentById.andCallFake(function() {
                        return {
                            then: function(successCb, failCb) {
                                failCb(); 
                            }
                        }
                    });
                    inject(function($controller, $rootScope) {
                        scope = $rootScope.$new();
                        controller = $controller('AssessmentBuilderEditController', {$scope:scope, $stateParams: {id: '123', assessment: null}});
                    });
                    expect(stateMock.go).toHaveBeenCalled(); 
                });
            });

            it("should create a blank assessment if there are no state params", function() {
                inject(function($controller, $rootScope) {
                    scope = $rootScope.$new();
                    controller = $controller('AssessmentBuilderEditController', {$scope:scope, $stateParams: {}});
                });
                expect(scope.assessment.uniqueTitle).toBe(""); 
            }); 
        });

        describe("open instructions function", function() {
            it("should open a modal", function() {
                scope.openInstructionsModal();
                expect(modalMock.open).toHaveBeenCalled();
                expect(focusServiceMock.focusElement).toHaveBeenCalled(); 
            });
        });

        describe("open my VA Health information function", function() {
            it("should open a modal", function() {
                scope.openMyVAHealthModal();
                expect(modalServiceMock.showModal).toHaveBeenCalled(); 
                expect(focusServiceMock.focusElement).toHaveBeenCalled(); 
            });
        });

        describe("add question function", function() {
            it("should add a question and move focus to that question", function() {
                scope.assessment.question = [{}, {}, {}];
                scope.addQuestion();
                expect(scope.assessment.question.length).toBe(4);
                expect(focusServiceMock.focusElement).toHaveBeenCalled();
            }); 
        });

        describe("change question function", function() {
            it("should change the question type at a certain index", function() {
                scope.assessment.question = [{
                    questionType: undefined
                }, {
                    questionType: undefined
                }];
                scope.changeQuestion('slider', 0); 
                scope.changeQuestion('singleSelect', 1);
                expect(scope.assessment.question[0].questionType).toBe('slider');
                expect(scope.assessment.question[0].test).toBe('data'); 
                expect(scope.assessment.question[1].questionType).toBe('singleSelect');
                expect(scope.assessment.question[1].test).toBe('data'); 
                scope.changeQuestion(undefined, 0);
                expect(scope.assessment.question[0].questionType).toBe(undefined);
                expect(scope.assessment.question[0].test).toBe(undefined); 
            });
        });

        describe("remove question function", function() {
            it("should remove a question and return focus to add question button", function() {
                scope.assessment.question = [1, 2, 3, 4];
                scope.removeQuestion(1);
                expect(scope.assessment.question).toEqual([1, 3, 4]);
                expect(focusServiceMock.focusElement).toHaveBeenCalled();
            }); 
        });

        describe("update slider at index", function() {
            it("should update the maxvalue value of the slider", function() {
                scope.assessment.question = [{
                    questionProperties: [{
                        name: 'minValue',
                        value: '1'
                    }, {
                        name: 'maxValue',
                        value: '5'
                    }]
                }];
                scope.updateSliderAtIndex(0, '10'); 
                expect(scope.assessment.question[0].questionProperties[1].value).toBe('10'); 
            });
        });

        describe("delete function", function() {
            it("should warn about deleting", function() {
                scope.delete(); 
                expect(stateMock.go).toHaveBeenCalled(); 
                expect(focusServiceMock.focusElement).toHaveBeenCalled();
            });
        });

        describe("save function", function() {
            var formCtrl;

            beforeEach(function() {
                formCtrl = {
                    validationSummary: {
                        validate: function(){}
                    },
                    $setPristine: function(){}
                };
                spyOn(formCtrl.validationSummary, 'validate');
            }); 

            afterEach(function() {
                expect(formCtrl.validationSummary.validate).toHaveBeenCalled();
            });

            it("should not do anything if the form is not valid", function() {
                formCtrl.$valid = false;
                scope.save(formCtrl);
                expect(assessmentServiceMock.saveAssessment).not.toHaveBeenCalled();
            });
            
            it("should handle a successful save", function() {
                formCtrl.$valid = true;
                scope.save(formCtrl);
                expect(assessmentServiceMock.saveAssessment).toHaveBeenCalled();
                expect(scope.assessment).toEqual("good response"); 
                expect(modalServiceMock.showModal).toHaveBeenCalled(); 
                expect(focusServiceMock.focusElement).toHaveBeenCalled(); 
            });

            it("should append the id to the url if it is not there", function() {
                inject(function($controller, $rootScope) {
                    scope = $rootScope.$new();
                    controller = $controller('AssessmentBuilderEditController', {$scope:scope, $stateParams: {}});
                });
                expect(scope.assessment.uniqueTitle).toBe(""); 
                formCtrl.$valid = true;
                scope.save(formCtrl);
                expect(locationMock.path).toHaveBeenCalled(); 
            });

            it("should handle a duplicate record", function() {
                assessmentServiceMock.saveAssessment.andCallFake(function() {
                    return {
                        then: function(successCb, failCb) {
                            failCb({data: {errorMessage: "record exists"}});
                        }
                    };
                });
                scope.assessment.uniqueTitle = "abc";
                formCtrl.$valid = true;
                scope.save(formCtrl);
                expect(assessmentServiceMock.saveAssessment).toHaveBeenCalled();
                expect(scope.titleRegex.toString()).toEqual("/^(?!\\s*abc\\s*$).*$/");
            });

            it("should handle other errors", function() {
                assessmentServiceMock.saveAssessment.andCallFake(function() {
                    return {
                        then: function(successCb, failCb) {
                            failCb({data: {errorMessage: "another error"}});
                        }
                    };
                });
                formCtrl.$valid = true;
                scope.save(formCtrl);
                expect(assessmentServiceMock.saveAssessment).toHaveBeenCalled();
                expect(scope.titleRegex.toString()).toEqual("/^.*$/");
            });
        });

        describe("publish function", function() {
            var formCtrl;

            beforeEach(function() {
                scope.assessment.question = ['A question']; 
                formCtrl = {
                    validationSummary: {
                        validate: function(){
                            return {
                                then: function(successCb, errorCb) {
                                    successCb();
                                    errorCb(); 
                                }
                            }
                        },
                        pushError: function(){}
                    },
                    $setPristine: function(){}
                };
                spyOn(formCtrl.validationSummary, 'pushError');
            }); 

            it("should push an error if there is not a question in the assessment", function() {
                scope.assessment.question = [];
                formCtrl.$valid = true;
                scope.publish(formCtrl);
                expect(assessmentServiceMock.saveAssessment).not.toHaveBeenCalled();
                expect(formCtrl.validationSummary.pushError).toHaveBeenCalled(); 
            });

            it("should not push an error if there is a question in the assessment", function() {
                formCtrl.$valid = false;
                scope.publish(formCtrl);
                expect(formCtrl.validationSummary.pushError).not.toHaveBeenCalled(); 
            });

            it("should not do anything if the form is not valid", function() {
                formCtrl.$valid = false;
                scope.publish(formCtrl);
                expect(assessmentServiceMock.saveAssessment).not.toHaveBeenCalled();
            });

            it("should warn if the form is dirty", function() {
                formCtrl.$valid = true;
                formCtrl.$dirty = true;
                scope.publish(formCtrl);
                expect(modalServiceMock.showModal).toHaveBeenCalled(); 
            });

            it("should go straight to the publish prompt if the form is not dirty", function() {
                formCtrl.$valid = true;
                formCtrl.$dirty = false;
                scope.publish(formCtrl);
                expect(scope.assessment.publish).toBe(true); 
                expect(modalServiceMock.showModal).toHaveBeenCalled(); 
            });
        });

        describe("inactivate function", function() {
            it("should inactivate the assessment", function() {
                scope.inactivate();
                expect(assessmentServiceMock.inactivateAssessment).toHaveBeenCalled(); 
                expect(stateMock.go).toHaveBeenCalled();
                expect(focusServiceMock.focusElement).toHaveBeenCalled(); 
            });
        });

        describe("modify function", function() {
            it("should open the assessment to modifications", function() {
                scope.modify();
                expect(assessmentServiceMock.modifyAssessment).toHaveBeenCalled(); 
                expect(stateMock.go).toHaveBeenCalled();
                expect(focusServiceMock.focusElement).toHaveBeenCalled(); 
            });
        });
    });
});